import Users from "../models/userModel.js";
import DataPeserta from "../models/dataPesertaModel.js";
import DataEvent from "../models/dataEventModel.js";
import DataBurung from "../models/dataBurungModel.js";
import db from "../config/database.js";
import { Op } from "sequelize";

export const getProfile = async (req, res) => {
  try {
    const user = await Users.findOne({
      where: { no_hp: req.user.nohp },
      attributes: ["id", "name", "no_hp", "role"],
      include: [
        {
          model: DataPeserta,
          as: "data_peserta",
          required: false,
        },
      ],
    });

    if (!user) {
      return res.status(404).json({ msg: "Pengguna tidak ditemukan" });
    }

    // DITINGKATKAN: Logika saat DataPeserta belum ada
    if (!user.data_peserta) {
      // Langsung buat DataPeserta dengan nama yang sudah sinkron dari tabel Users
      const newDataPeserta = await DataPeserta.create({
        no_hp: user.no_hp,
        name: user.name, // <-- SINKRONISASI: `name` diisi
        nama_peserta: user.name, // <-- SINKRONISASI: `nama_peserta` diisi dengan `name` user
        nama_kandang: "",
        alamat: "",
        kota_kabupaten: "",
        provinsi: "",
        club: "",
        latitude: null,
        longitude: null,
      });
      // Langsung gabungkan data baru ke objek user untuk respons
      user.data_peserta = newDataPeserta;
    }

    const isCoordinateLocked =
      user.data_peserta.latitude != null && user.data_peserta.longitude != null;

    // DITINGKATKAN: Respons JSON dengan fallback untuk memastikan nama_peserta selalu ada
    res.json({
      id: user.id,
      name: user.name,
      phone_number: user.no_hp,
      role: user.role,
      nama_peserta: user.data_peserta?.nama_peserta || user.name || "",
      nama_kandang: user.data_peserta?.nama_kandang || "",
      alamat: user.data_peserta?.alamat || "",
      kota_kabupaten: user.data_peserta?.kota_kabupaten || "",
      provinsi: user.data_peserta?.provinsi || "",
      club: user.data_peserta?.club || "",
      latitude: user.data_peserta?.latitude || "",
      longitude: user.data_peserta?.longitude || "",
      is_coordinate_locked: isCoordinateLocked,
    });
  } catch (error) {
    console.error("Error saat mengambil profil:", error.message);
    res
      .status(500)
      .json({ msg: "Kesalahan Server Internal", error: error.message });
  }
};
export const getUserProfile = async (req, res) => {
  try {
    // Pastikan hanya superadmin yang dapat mengakses
    if (req.user.role !== "superadmin") {
      return res
        .status(403)
        .json({ msg: "Akses ditolak, hanya untuk superadmin" });
    }

    const { id } = req.params;
    const user = await Users.findOne({
      where: { id },
      attributes: ["id", "name", "no_hp", "role"],
      include: [
        {
          model: DataPeserta,
          attributes: [
            "nama_peserta",
            "nama_kandang",
            "alamat",
            "kota_kabupaten",
            "provinsi",
            "club",
            "latitude",
            "longitude",
          ],
          required: false,
          as: "data_peserta",
        },
      ],
    });

    if (!user) {
      return res.status(404).json({ msg: "Pengguna tidak ditemukan" });
    }

    if (!user.data_peserta) {
      await DataPeserta.create({
        no_hp: user.no_hp,
        nama_peserta: "",
        nama_kandang: "",
        alamat: "",
        kota_kabupaten: "",
        provinsi: "",
        club: "",
        latitude: null,
        longitude: null,
      });
      user.data_peserta = await DataPeserta.findOne({
        where: { no_hp: user.no_hp },
      });
    }

    const isCoordinateLocked =
      user.data_peserta.latitude != null && user.data_peserta.longitude != null;

    res.json({
      id: user.id,
      name: user.name,
      phone_number: user.no_hp,
      role: user.role,
      nama_peserta: user.data_peserta?.nama_peserta || "",
      nama_kandang: user.data_peserta?.nama_kandang || "",
      alamat: user.data_peserta?.alamat || "",
      kota_kabupaten: user.data_peserta?.kota_kabupaten || "",
      provinsi: user.data_peserta?.provinsi || "",
      club: user.data_peserta?.club || "",
      latitude: user.data_peserta?.latitude || "",
      longitude: user.data_peserta?.longitude || "",
      is_coordinate_locked: isCoordinateLocked,
    });
  } catch (error) {
    console.error("Error saat mengambil profil pengguna:", error.message);
    res
      .status(500)
      .json({ msg: "Kesalahan Server Internal", error: error.message });
  }
};
export const adminDeleteUser = async (req, res) => {
  const { id } = req.params;

  try {
    const user = await Users.findByPk(id);

    if (!user) {
      return res.status(404).json({ msg: "Pengguna tidak ditemukan." });
    }

    // Dengan 'onDelete: CASCADE' di model, baris ini saja sudah cukup
    // untuk menghapus user, data_peserta, dan semua data_burung terkait.
    await user.destroy();

    res
      .status(200)
      .json({
        msg: `Pengguna dengan ID ${id} dan semua data terkait berhasil dihapus.`,
      });
  } catch (error) {
    console.error("Error saat menghapus pengguna:", error);
    res
      .status(500)
      .json({ msg: "Terjadi kesalahan pada server", error: error.message });
  }
};
export const updateProfile = async (req, res) => {
  const t = await db.transaction(); // DITINGKATKAN: Menggunakan transaksi untuk keamanan data

  try {
    if (!req.user || !req.user.nohp) {
      await t.rollback();
      return res.status(401).json({ msg: "Autentikasi gagal" });
    }

    const {
      nama_peserta,
      nama_kandang,
      phone_number,
      alamat,
      kota_kabupaten,
      provinsi,
      club,
      latitude,
      longitude,
    } = req.body;

    const user = await Users.findOne({
      where: { no_hp: req.user.nohp },
      include: [{ model: DataPeserta, as: "data_peserta" }],
      transaction: t,
    });

    if (!user) {
      await t.rollback();
      return res.status(404).json({ msg: "Pengguna tidak ditemukan" });
    }

    // --- SINKRONISASI INTI ---
    // Memperbarui `name` di tabel Users dan `nama_peserta` di tabel DataPeserta
    await user.update(
      {
        name: nama_peserta, // <-- INI BAGIAN PENTING: `name` diupdate dari `nama_peserta`
        no_hp: phone_number || user.no_hp,
      },
      { transaction: t }
    );

    // Logika untuk menangani koordinat terkunci (dipertahankan dari kode Anda)
    const isCoordinateLocked =
      user.data_peserta &&
      user.data_peserta.latitude != null &&
      user.data_peserta.longitude != null;

    let lat = latitude && latitude.trim() !== "" ? parseFloat(latitude) : null;
    let lon =
      longitude && longitude.trim() !== "" ? parseFloat(longitude) : null;
    if (isCoordinateLocked && user.role !== "superadmin") {
      lat = user.data_peserta.latitude;
      lon = user.data_peserta.longitude;
    }

    // DITINGKATKAN: Update atau Buat DataPeserta dengan lebih ringkas
    const [dataPeserta, created] = await DataPeserta.findOrCreate({
      where: { no_hp: user.no_hp },
      defaults: {
        name: nama_peserta, // <-- SINKRONISASI
        nama_peserta: nama_peserta, // <-- SINKRONISASI
        nama_kandang,
        alamat,
        kota_kabupaten,
        provinsi,
        club,
        latitude: lat,
        longitude: lon,
      },
      transaction: t,
    });

    if (!created) {
      // Jika sudah ada, update datanya
      await dataPeserta.update(
        {
          name: nama_peserta, // <-- SINKRONISASI
          nama_peserta: nama_peserta, // <-- SINKRONISASI
          nama_kandang,
          alamat,
          kota_kabupaten,
          provinsi,
          club,
          latitude: lat,
          longitude: lon,
        },
        { transaction: t }
      );
    }

    await t.commit(); // Menyimpan semua perubahan jika berhasil

    // Mengambil data terbaru setelah commit untuk dikirim sebagai respons
    const updatedUser = await Users.findOne({
      where: { id: user.id },
      attributes: ["id", "name", "no_hp", "role"],
      include: [{ model: DataPeserta, as: "data_peserta" }],
    });

    const updatedCoordinateLocked =
      updatedUser.data_peserta.latitude != null &&
      updatedUser.data_peserta.longitude != null;

    res.json({
      id: updatedUser.id,
      name: updatedUser.name,
      phone_number: updatedUser.no_hp,
      role: updatedUser.role,
      nama_peserta: updatedUser.data_peserta?.nama_peserta || "",
      nama_kandang: updatedUser.data_peserta?.nama_kandang || "",
      alamat: updatedUser.data_peserta?.alamat || "",
      kota_kabupaten: updatedUser.data_peserta?.kota_kabupaten || "",
      provinsi: updatedUser.data_peserta?.provinsi || "",
      club: updatedUser.data_peserta?.club || "",
      latitude: updatedUser.data_peserta?.latitude || "",
      longitude: updatedUser.data_peserta?.longitude || "",
      is_coordinate_locked: updatedCoordinateLocked,
    });
  } catch (error) {
    await t.rollback(); // Batalkan semua perubahan jika terjadi error
    console.error("Kesalahan saat memperbarui profil:", error.message);
    res
      .status(500)
      .json({ msg: "Kesalahan Server Internal", error: error.message });
  }
};

export const getEvents = async (req, res) => {
  try {
    const { page = 1, limit = 10, search = "" } = req.query;
    const offset = (page - 1) * limit;

    const where = search
      ? {
          [Op.or]: [
            { event_name: { [Op.like]: `%${search}%` } },
            { season: { [Op.like]: `%${search}%` } },
            { club_name: { [Op.like]: `%${search}%` } },
          ],
        }
      : {};

    const { count, rows } = await DataEvent.findAndCountAll({
      where,
      attributes: [
        "id",
        "event_name",
        "season",
        "club_name",
        "time_schedule",
        "race_count",
        "createdAt",
      ],
      order: [["createdAt", "DESC"]],
      limit: parseInt(limit),
      offset: parseInt(offset),
    });

    // Ubah nama field createdAt menjadi created_at untuk sesuai dengan frontend
    const events = rows.map((event) => ({
      id: event.id,
      event_name: event.event_name,
      season: event.season,
      club_name: event.club_name,
      time_schedule: event.time_schedule,
      race_count: event.race_count,
      created_at: event.createdAt,
    }));

    res.status(200).json({
      events,
      total: count,
      page: parseInt(page),
      limit: parseInt(limit),
    });
  } catch (error) {
    console.error("Error fetching events:", error);
    res.status(500).json({ message: "Gagal mengambil data event" });
  }
};

export const getEvent = async (req, res) => {
  try {
    const { id } = req.params;
    console.log("Mengambil event dengan ID:", id);
    const event = await DataEvent.findOne({
      where: { id },
      attributes: [
        "id",
        "event_name",
        "season",
        "club_id",
        "club_name",
        "time_schedule",
        "race_status",
        "race_count",
        "ranking_total_pos",
        "radius_clock",
        "createdAt",
      ],
    });

    if (!event) {
      console.log("Event tidak ditemukan untuk ID:", id);
      return res.status(404).json({ msg: "Event tidak ditemukan" });
    }

    // Ubah nama field createdAt menjadi created_at untuk sesuai dengan frontend
    res.status(200).json({
      id: event.id,
      event_name: event.event_name,
      season: event.season,
      club_id: event.club_id,
      club_name: event.club_name,
      time_schedule: event.time_schedule,
      race_status: event.race_status,
      race_count: event.race_count,
      ranking_total_pos: event.ranking_total_pos,
      radius_clock: event.radius_clock,
      created_at: event.createdAt,
    });
  } catch (error) {
    console.error("Kesalahan mengambil event:", error.message, error.stack);
    res.status(500).json({ msg: "Terjadi kesalahan pada server" });
  }
};

export const createEvent = async (req, res) => {
  try {
    const {
      event_name,
      season,
      club_id,
      club_name,
      time_schedule,
      race_status,
      race_count,
      ranking_total_pos,
      radius_clock,
    } = req.body;

    const event = await DataEvent.create({
      event_name,
      season,
      club_id,
      club_name,
      time_schedule,
      race_status,
      race_count: parseInt(race_count),
      ranking_total_pos: ranking_total_pos ? parseInt(ranking_total_pos) : null,
      radius_clock: radius_clock ? parseFloat(radius_clock) : null,
    });

    res.json({ msg: "Event created successfully", event });
  } catch (error) {
    console.log(error);
    res.status(500).json({ msg: "Internal Server Error" });
  }
};

export const updateEvent = async (req, res) => {
  try {
    const { id } = req.params;
    const {
      event_name,
      season,
      club_id,
      club_name,
      time_schedule,
      race_status,
      race_count,
      ranking_total_pos,
      radius_clock,
    } = req.body;

    const event = await DataEvent.findByPk(id);
    if (!event) return res.status(404).json({ msg: "Event not found" });

    await event.update({
      event_name,
      season,
      club_id,
      club_name,
      time_schedule,
      race_status,
      race_count: parseInt(race_count),
      ranking_total_pos: ranking_total_pos ? parseInt(ranking_total_pos) : null,
      radius_clock: radius_clock ? parseFloat(radius_clock) : null,
    });

    res.json({ msg: "Event updated successfully" });
  } catch (error) {
    console.log(error);
    res.status(500).json({ msg: "Internal Server Error" });
  }
};

export const deleteEvent = async (req, res) => {
  try {
    const { id } = req.params;
    const event = await DataEvent.findByPk(id);
    if (!event) return res.status(404).json({ msg: "Event not found" });

    await event.destroy();
    res.json({ msg: "Event deleted successfully" });
  } catch (error) {
    console.log(error);
    res.status(500).json({ msg: "Internal Server Error" });
  }
};

export const getBurung = async (req, res) => {
  try {
    const user = await Users.findOne({ where: { no_hp: req.user.nohp } });
    if (!user) {
      return res.status(404).json({ msg: "User not found" });
    }

    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const search = req.query.search || ""; // Parameter untuk pencarian nama burung
    const offset = (page - 1) * limit;

    let whereClause = { user_id: user.id }; // Filter berdasarkan user_id tetap ada

    if (search) {
      whereClause.bird_name = {
        // Tambahkan kondisi pencarian pada bird_name
        [Op.like]: `%${search}%`,
      };
    }

    const { count, rows: burungData } = await DataBurung.findAndCountAll({
      where: whereClause,
      attributes: ["id", "bird_name", "COL", "SEX", "createdAt"], // Ambil createdAt untuk sorting jika perlu
      limit: limit,
      offset: offset,
      order: [["createdAt", "DESC"]], // Contoh sorting, bisa disesuaikan
    });

    res.json({
      // Kembalikan data dengan format pagination
      data: burungData,
      total: count,
      page: page,
      limit: limit,
      totalPages: Math.ceil(count / limit),
    });
  } catch (error) {
    console.log(error); // Sebaiknya error.message untuk detail lebih baik
    res.status(500).json({ msg: "Internal Server Error" });
  }
};
export const getAllBirdsWithOwners = async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const search = req.query.search || "";
    const offset = (page - 1) * limit;

    let whereClause = {}; // Kondisi pencarian utama untuk DataBurung

    // Kondisi untuk include Users, akan diterapkan jika search tidak kosong
    // untuk memungkinkan pencarian berdasarkan nama user juga.
    // Perhatikan bahwa '$user.name$' merujuk pada kolom 'name' dari model 'Users'
    // yang di-alias sebagai 'user' dalam include.
    // Pastikan alias 'user' konsisten dengan definisi relasi Anda.
    // Di dataBurungModel.js: DataBurung.belongsTo(Users, { foreignKey: "user_id" });
    // Sequelize secara default bisa menggunakan nama model lowercase ('user') atau nama model itu sendiri ('User')
    // Untuk kepastian, Anda bisa menambahkan `as: 'user'` pada relasi belongsTo dan include.
    // Kita asumsikan 'user' adalah alias yang benar untuk saat ini.

    if (search) {
      whereClause = {
        [Op.or]: [
          { bird_name: { [Op.like]: `%${search}%` } },
          // PENTING: '$user.name$' memerlukan alias 'user' pada include Users
          // atau sesuaikan dengan nama alias yang benar jika berbeda.
          // Jika tidak ada alias eksplisit, coba '$User.name$' (nama model)
          { "$user.name$": { [Op.like]: `%${search}%` } },
        ],
      };
    }

    const { count, rows: allBirdsData } = await DataBurung.findAndCountAll({
      where: whereClause,
      attributes: ["id", "bird_name", "COL", "SEX", "createdAt"],
      include: [
        {
          model: Users,
          as: "user", // <-- Tambahkan alias 'user' di sini agar '$user.name$' berfungsi
          attributes: ["name"],
          required: true, // Tetap true agar hanya burung dengan pemilik yang valid ditampilkan
        },
      ],
      order: [["createdAt", "DESC"]],
      limit: limit,
      offset: offset,
      distinct: true, // Tambahkan distinct true karena pencarian dengan Op.or pada include bisa menyebabkan duplikasi row sebelum dihitung.
      subQuery: false, // Penting jika 'where' pada include menyebabkan masalah dengan limit/offset/count
      // Mungkin perlu disesuaikan jika count tidak akurat.
      // Alternatifnya adalah melakukan count terpisah jika subQuery:false tidak menyelesaikan.
    });

    const formattedBirds = allBirdsData.map((bird) => ({
      id: bird.id,
      nama_pemilik: bird.user ? bird.user.name : "Pemilik Tidak Diketahui",
      bird_name: bird.bird_name,
      COL: bird.COL,
      SEX: bird.SEX,
    }));

    res.status(200).json({
      data: formattedBirds,
      total: count, // count dari findAndCountAll mungkin perlu penyesuaian jika ada subQuery:false atau duplikasi
      page: page,
      limit: limit,
      totalPages: Math.ceil(count / limit),
    });
  } catch (error) {
    console.error(
      "Error fetching all birds with owners:",
      error.message,
      error.stack
    );
    res
      .status(500)
      .json({ msg: "Gagal mengambil data semua burung", error: error.message });
  }
};

export const getUserDetailsForAdminEdit = async (req, res) => {
  try {
    // Pastikan hanya superadmin yang dapat mengakses
    if (req.user.role !== "superadmin") {
      return res
        .status(403)
        .json({ msg: "Akses ditolak, hanya untuk superadmin" });
    }

    const { id } = req.params;
    const user = await Users.findOne({
      where: { id: id },
      attributes: ["id", "name", "no_hp", "role"],
      include: [
        {
          model: DataPeserta,
          attributes: [
            "nama_peserta",
            "nama_kandang",
            "alamat",
            "kota_kabupaten",
            "provinsi",
            "club",
            "latitude",
            "longitude",
          ],
          required: false, // DataPeserta mungkin belum ada, jadi false
          as: "data_peserta", // Sesuai dengan alias di model
        },
        {
          model: DataBurung,
          attributes: ["id", "bird_name", "COL", "SEX"],
          required: false, // User mungkin belum punya burung
          // Jika Anda mendefinisikan alias 'as' di Users.hasMany(DataBurung, {as: 'data_burung_user'}), gunakan di sini
          // Jika tidak, Sequelize biasanya akan menggunakan nama jamak dari model (data_burungs) atau nama model.
          // Untuk memastikan, Anda bisa menambahkan alias di relasi Users.hasMany(DataBurung, { foreignKey: "user_id", as: "data_burung" });
          // dan gunakan as: "data_burung" di sini.
          // Jika tidak ada alias, Sequelize akan mencoba mengambilnya.
          // Berdasarkan model Anda Users.hasMany(DataBurung, { foreignKey: "user_id" });
          // Sequelize akan otomatis menamakannya 'data_burungs' jika tidak ada alias, atau Anda bisa langsung menggunakan model DataBurung.
        },
      ],
    });

    if (!user) {
      return res.status(404).json({ msg: "Pengguna tidak ditemukan" });
    }

    // Jika DataPeserta belum ada, buat entri kosong (opsional, tergantung kebutuhan)
    // Kode ini mirip dengan yang ada di getUserProfile, Anda bisa menyesuaikannya.
    // Jika Anda selalu ingin data_peserta ada (meski kosong), logika ini bisa dipertahankan.
    let dataPesertaResponse = user.data_peserta;
    if (!user.data_peserta) {
      // Anda bisa memilih untuk tidak membuat data peserta default di sini
      // dan membiarkannya null jika memang tidak ada,
      // frontend akan menangani null tersebut.
      // Atau, jika Anda ingin konsisten membuatkannya:
      // const newPeserta = await DataPeserta.create({ no_hp: user.no_hp /* field lainnya default null/kosong */ });
      // dataPesertaResponse = newPeserta.toJSON(); // atau setidaknya field-field yang dibutuhkan
      dataPesertaResponse = null; // Biarkan null jika tidak ada, frontend akan handle
    }

    // Menggunakan user.data_burungs jika tidak ada alias 'as' spesifik yang konsisten digunakan,
    // atau nama alias yang Anda definisikan di relasi Users.hasMany(DataBurung, {...}).
    // Untuk lebih pasti, selalu lebih baik menggunakan alias eksplisit di kedua sisi (model dan query).
    // Misal jika di userModel.js: Users.hasMany(DataBurung, { foreignKey: 'user_id', as: 'burungPengguna' });
    // Maka di sini user.burungPengguna

    // Asumsi Sequelize secara default menamakannya 'data_burungs' (plural dari DataBurung)
    // atau jika Anda telah menambahkan alias `as: 'data_burung'` pada `Users.hasMany` di userModel.js,
    // maka Anda bisa menggunakan `user.data_burung`
    const dataBurungResponse = user.data_burungs || user.DataBurungs || []; // Sesuaikan dengan bagaimana Sequelize mengembalikan relasi hasMany Anda.
    // Jika Anda menggunakan alias di Users.hasMany(DataBurung, {as: 'birds'}), maka user.birds

    res.json({
      id: user.id,
      name: user.name,
      no_hp: user.no_hp,
      role: user.role,
      data_peserta: dataPesertaResponse
        ? {
            // Kirim null jika dataPesertaResponse null
            nama_peserta: dataPesertaResponse.nama_peserta || "",
            nama_kandang: dataPesertaResponse.nama_kandang || "",
            alamat: dataPesertaResponse.alamat || "",
            kota_kabupaten: dataPesertaResponse.kota_kabupaten || "",
            provinsi: dataPesertaResponse.provinsi || "",
            club: dataPesertaResponse.club || "",
            latitude: dataPesertaResponse.latitude, // Biarkan null jika memang null
            longitude: dataPesertaResponse.longitude, // Biarkan null jika memang null
          }
        : null,
      // data_burung akan menjadi array dari objek burung
      data_burung: dataBurungResponse.map((b) => ({
        id: b.id,
        bird_name: b.bird_name,
        COL: b.COL,
        SEX: b.SEX,
      })),
    });
  } catch (error) {
    console.error(
      "Error saat mengambil detail pengguna untuk admin edit:",
      error.message
    );
    res
      .status(500)
      .json({ msg: "Kesalahan Server Internal", error: error.message });
  }
};
export const adminUpdateUser = async (req, res) => {
  // Pastikan hanya superadmin yang dapat mengakses
  if (req.user.role !== "superadmin") {
    return res
      .status(403)
      .json({ msg: "Akses ditolak, hanya untuk superadmin" });
  }

  const { id: userIdToUpdate } = req.params;
  const { name, no_hp, password, data_peserta, birds } = req.body;

  const t = await db.transaction(); // Mulai transaksi

  try {
    // 1. Temukan Pengguna yang akan diupdate
    const user = await Users.findByPk(userIdToUpdate, { transaction: t });
    if (!user) {
      await t.rollback();
      return res.status(404).json({ msg: "Pengguna tidak ditemukan" });
    }

    const oldNoHp = user.no_hp;

    // 2. Update Data Utama Pengguna (Users table)
    const userDataToUpdate = {
      name: name || user.name,
      no_hp: no_hp || user.no_hp,
    };
    if (password) {
      const salt = await bcrypt.genSalt();
      userDataToUpdate.password = await bcrypt.hash(password, salt);
    }
    await user.update(userDataToUpdate, { transaction: t });

    // 3. Update atau Buat Data Peserta (DataPeserta table)
    if (data_peserta) {
      const pesertaData = {
        nama_peserta: data_peserta.nama_peserta,
        nama_kandang: data_peserta.nama_kandang,
        alamat: data_peserta.alamat,
        kota_kabupaten: data_peserta.kota_kabupaten,
        provinsi: data_peserta.provinsi,
        club: data_peserta.club,
        latitude: data_peserta.latitude,
        longitude: data_peserta.longitude,
        no_hp: user.no_hp, // Gunakan no_hp yang mungkin baru dari user
      };

      // Jika no_hp user diubah, kita perlu menangani DataPeserta yang terkait dengan no_hp lama
      if (oldNoHp !== user.no_hp) {
        const existingDataPesertaOldHp = await DataPeserta.findOne({
          where: { no_hp: oldNoHp },
          transaction: t,
        });
        if (existingDataPesertaOldHp) {
          // Pilihan 1: Update no_hp di DataPeserta yang lama
          await existingDataPesertaOldHp.update(
            { no_hp: user.no_hp, ...pesertaData },
            { transaction: t }
          );
        } else {
          // Pilihan 2: Atau buat baru jika tidak ada yang lama dan coba update/create dengan no_hp baru
          await DataPeserta.upsert(pesertaData, {
            where: { no_hp: user.no_hp }, // Ini tidak didukung langsung oleh upsert, upsert menggunakan primary key atau unique constraint
            transaction: t,
          });
          // Cara yang lebih aman untuk upsert berdasarkan no_hp:
          let currentPeserta = await DataPeserta.findOne({
            where: { no_hp: user.no_hp },
            transaction: t,
          });
          if (currentPeserta) {
            await currentPeserta.update(pesertaData, { transaction: t });
          } else {
            await DataPeserta.create(pesertaData, { transaction: t });
          }
        }
      } else {
        // no_hp tidak berubah, cukup update/create berdasarkan no_hp saat ini
        let currentPeserta = await DataPeserta.findOne({
          where: { no_hp: user.no_hp },
          transaction: t,
        });
        if (currentPeserta) {
          await currentPeserta.update(pesertaData, { transaction: t });
        } else {
          await DataPeserta.create(pesertaData, { transaction: t });
        }
      }
    }

    // 4. Update Data Burung (DataBurung table)
    if (Array.isArray(birds)) {
      const existingBirdIds = birds.filter((b) => b.id).map((b) => b.id);

      // Hapus burung yang tidak ada di payload tapi ada di DB untuk user ini
      await DataBurung.destroy({
        where: {
          user_id: user.id,
          id: { [Op.notIn]: existingBirdIds }, // Hanya jika existingBirdIds tidak kosong
        },
        transaction: t,
      });

      // Jika existingBirdIds kosong dan birds juga kosong, berarti semua burung dihapus
      if (existingBirdIds.length === 0 && birds.length === 0) {
        await DataBurung.destroy({
          where: { user_id: user.id },
          transaction: t,
        });
      }

      for (const birdData of birds) {
        if (birdData.id) {
          // Update burung yang sudah ada
          const burungToUpdate = await DataBurung.findOne({
            where: { id: birdData.id, user_id: user.id },
            transaction: t,
          });
          if (burungToUpdate) {
            await burungToUpdate.update(
              {
                bird_name: birdData.bird_name,
                COL: birdData.COL,
                SEX: birdData.SEX,
              },
              { transaction: t }
            );
          }
        } else {
          // Buat burung baru
          await DataBurung.create(
            {
              user_id: user.id,
              bird_name: birdData.bird_name,
              COL: birdData.COL,
              SEX: birdData.SEX,
            },
            { transaction: t }
          );
        }
      }
    }

    await t.commit(); // Commit transaksi jika semua berhasil
    res.json({ msg: "Data pengguna berhasil diperbarui" });
  } catch (error) {
    await t.rollback(); // Rollback transaksi jika terjadi error
    console.error(
      "Error saat admin memperbarui profil:",
      error.message,
      error.stack
    );
    res
      .status(500)
      .json({ msg: "Kesalahan Server Internal", error: error.message });
  }
};
export const createBurung = async (req, res) => {
  try {
    const { bird_name, COL, SEX } = req.body;
    const user = await Users.findOne({ where: { no_hp: req.user.nohp } });
    if (!user) return res.status(404).json({ msg: "User not found" });

    const burung = await DataBurung.create({
      user_id: user.id,
      bird_name,
      COL,
      SEX,
    });

    res.json({ msg: "Bird added successfully", burung });
  } catch (error) {
    console.log(error);
    res.status(500).json({ msg: "Internal Server Error" });
  }
};

export const deleteBurung = async (req, res) => {
  try {
    const { id } = req.params;
    const burung = await DataBurung.findByPk(id);
    if (!burung) return res.status(404).json({ msg: "Bird not found" });

    await burung.destroy();
    res.json({ msg: "Bird deleted successfully" });
  } catch (error) {
    console.log(error);
    res.status(500).json({ msg: "Internal Server Error" });
  }
};

// Fungsi baru untuk mendaftarkan burung ke event
export const registerBirdToEvent = async (req, res) => {
  const { eventId, birdId } = req.body;
  const userId = req.userId;

  if (!eventId || !birdId || !userId) {
    return res
      .status(400)
      .json({ msg: "Event ID, Bird ID, dan User ID diperlukan!" });
  }

  try {
    const event = await DataEvent.findByPk(eventId);
    if (!event) {
      return res.status(404).json({ msg: "Event tidak ditemukan." });
    }

    const bird = await DataBurung.findOne({
      where: {
        id: birdId,
        user_id: userId,
      },
    });
    if (!bird) {
      return res
        .status(404)
        .json({ msg: "Burung tidak ditemukan atau bukan milik Anda." });
    }

    const racesInEvent = await Race.findAll({
      where: {
        event_id: eventId,
      },
    });

    if (racesInEvent.length === 0) {
      return res.status(404).json({
        msg: "Tidak ada race yang ditemukan untuk event ini. Pastikan admin telah menambahkan race.",
      });
    }

    const registrations = [];
    let existingRegistrationsCount = 0;

    for (const race of racesInEvent) {
      try {
        const [registration, created] = await RaceRegistration.findOrCreate({
          where: {
            race_id: race.id,
            bird_id: birdId,
          },
          defaults: {
            bird_name: bird.bird_name, // TAMBAH INI: Simpan nama burung
            user_id: userId,
            status: "registered",
          },
        });

        if (created) {
          registrations.push(registration);
        } else {
          existingRegistrationsCount++;
        }
      } catch (error) {
        if (error.name === "SequelizeUniqueConstraintError") {
          console.warn(`Burung ${birdId} sudah terdaftar di Race ${race.id}.`);
          existingRegistrationsCount++;
        } else {
          console.error(
            `Gagal mendaftarkan burung ${birdId} ke Race ${race.id}:`,
            error.message
          );
          throw error;
        }
      }
    }

    if (registrations.length > 0) {
      return res.status(201).json({
        msg: `Burung "${bird.bird_name}" berhasil didaftarkan ke ${registrations.length} race dalam event "${event.event_name}".`,
        newRegistrations: registrations.length,
        existingRegistrations: existingRegistrationsCount,
      });
    } else if (existingRegistrationsCount > 0) {
      return res.status(200).json({
        msg: `Burung "${bird.bird_name}" sudah terdaftar di semua race yang relevan dalam event "${event.event_name}".`,
        existingRegistrations: existingRegistrationsCount,
      });
    } else {
      return res.status(500).json({
        msg: "Gagal mendaftarkan burung ke event, tidak ada pendaftaran yang dibuat.",
      });
    }
  } catch (error) {
    console.error("Error during bird registration to event:", error);
    res.status(500).json({
      msg:
        error.message ||
        "Terjadi kesalahan server saat mendaftarkan burung ke event.",
    });
  }
};
