import Race from "../models/RaceModel.js";
import DataEvent from "../models/dataEventModel.js";
import { Op } from "sequelize";

// 1. Get All Races for a specific Event (GET /races/:eventId)
export const getRacesByEvent = async (req, res) => {
  try {
    const { eventId } = req.params;

    const event = await DataEvent.findByPk(eventId);
    if (!event) {
      return res.status(404).json({ msg: "Event tidak ditemukan." });
    }

    const races = await Race.findAll({
      where: {
        event_id: eventId,
      },
      order: [["race_number", "ASC"]],
    });
    res.status(200).json(races);
  } catch (error) {
    console.error("Error getting races by event:", error);
    res.status(500).json({ msg: error.message || "Terjadi kesalahan server." });
  }
};

// 2. Create a new Race (POST /races)
export const createRace = async (req, res) => {
  const {
    event_id,
    race_number,
    race_name,
    race_status,
    kota_race,
    kelas_race,
    rangking_etape,
    kecepatan_minimal,
    pinalti_basketing,
    pinalti_clock,
    tanggal_inkorf,
    tanggal_race,
    jam_lepas,
    jam_buka_clock,
    jam_tutup_clock,
    tanggal_tutup,
    jam_tutup_race,
    distance_km,
    description,
    latitude, // Added latitude
    longitude, // Added longitude
  } = req.body;

  try {
    if (!event_id || !race_number || !race_name) {
      return res.status(400).json({
        msg: "Field event_id, race_number, dan race_name wajib diisi.",
      });
    }

    const event = await DataEvent.findByPk(event_id);
    if (!event) {
      return res.status(404).json({ msg: "Event tidak ditemukan." });
    }

    const existingRacesCount = await Race.count({
      where: { event_id: event_id },
    });

    if (event.race_count !== null && existingRacesCount >= event.race_count) {
      return res.status(403).json({
        msg: `Jumlah race sudah mencapai batas maksimal (${event.race_count}) yang diatur di Event. Tidak bisa menambah race baru.`,
      });
    }

    const existingRace = await Race.findOne({
      where: {
        event_id: event_id,
        race_number: race_number,
      },
    });

    if (existingRace) {
      return res
        .status(409)
        .json({ msg: `Race ke-${race_number} sudah ada untuk event ini.` });
    }

    const newRace = await Race.create({
      event_id,
      race_number,
      race_name,
      race_status,
      kota_race,
      kelas_race,
      rangking_etape,
      kecepatan_minimal,
      pinalti_basketing,
      pinalti_clock,
      tanggal_inkorf,
      tanggal_race,
      jam_lepas,
      jam_buka_clock,
      jam_tutup_clock,
      tanggal_tutup,
      jam_tutup_race,
      distance_km: distance_km || null,
      description: description || null,
      latitude: latitude || null, // Added latitude
      longitude: longitude || null, // Added longitude
    });
    res.status(201).json({ msg: "Race berhasil ditambahkan!", race: newRace });
  } catch (error) {
    console.error("Error creating race:", error);
    res.status(500).json({
      msg: error.message || "Terjadi kesalahan server saat menambahkan race.",
    });
  }
};

// 3. Update a Race (PUT /races/:id)
export const updateRace = async (req, res) => {
  const { id } = req.params;
  const {
    // event_id, // event_id typically should not be changed on update, or handled carefully
    race_number,
    race_name,
    race_status,
    kota_race,
    kelas_race,
    rangking_etape,
    kecepatan_minimal,
    pinalti_basketing,
    pinalti_clock,
    tanggal_inkorf,
    tanggal_race,
    jam_lepas,
    jam_buka_clock,
    jam_tutup_clock,
    tanggal_tutup,
    jam_tutup_race,
    distance_km,
    description,
    latitude, // Added latitude
    longitude, // Added longitude
  } = req.body;

  try {
    const race = await Race.findByPk(id);
    if (!race) {
      return res.status(404).json({ msg: "Race tidak ditemukan." });
    }

    if (race_number && race_number !== race.race_number) {
      const existingRace = await Race.findOne({
        where: {
          event_id: race.event_id, // Use existing event_id from the race being updated
          race_number: race_number,
          id: { [Op.ne]: id },
        },
      });
      if (existingRace) {
        return res
          .status(409)
          .json({ msg: `Race ke-${race_number} sudah ada untuk event ini.` });
      }
    }

    // Prepare update data, only update fields that are provided in req.body
    const updateData = { ...req.body };
    // Remove event_id from updateData if present, as it's not typically updated this way
    // or ensure it matches race.event_id if included. For now, let's assume it's not changed.
    delete updateData.event_id;

    // Handle undefined for optional numeric fields to ensure they are set to null if explicitly cleared,
    // or retain existing value if not provided.
    updateData.distance_km =
      distance_km !== undefined
        ? distance_km === ""
          ? null
          : Number(distance_km)
        : race.distance_km;
    updateData.latitude =
      latitude !== undefined
        ? latitude === ""
          ? null
          : Number(latitude)
        : race.latitude;
    updateData.longitude =
      longitude !== undefined
        ? longitude === ""
          ? null
          : Number(longitude)
        : race.longitude;
    updateData.kecepatan_minimal =
      kecepatan_minimal !== undefined
        ? Number(kecepatan_minimal)
        : race.kecepatan_minimal;
    updateData.pinalti_basketing =
      pinalti_basketing !== undefined
        ? Number(pinalti_basketing)
        : race.pinalti_basketing;
    updateData.pinalti_clock =
      pinalti_clock !== undefined ? Number(pinalti_clock) : race.pinalti_clock;

    await race.update(updateData);

    res.status(200).json({ msg: "Race berhasil diperbarui!", race });
  } catch (error) {
    console.error("Error updating race:", error);
    res.status(500).json({
      msg: error.message || "Terjadi kesalahan server saat memperbarui race.",
    });
  }
};

// 4. Delete a Race (DELETE /races/:id)
export const deleteRace = async (req, res) => {
  try {
    const { id } = req.params;

    const race = await Race.findByPk(id);
    if (!race) {
      return res.status(404).json({ msg: "Race tidak ditemukan." });
    }

    await race.destroy();
    res.status(200).json({ msg: "Race berhasil dihapus." });
  } catch (error) {
    console.error("Error deleting race:", error);
    res.status(500).json({
      msg: error.message || "Terjadi kesalahan server saat menghapus race.",
    });
  }
};
