import Users from "../models/userModel.js";
import bcrypt from "bcrypt";
import jwt from "jsonwebtoken";
import DataPeserta from "../models/dataPesertaModel.js";

export const getUsers = async (req, res) => {
  try {
    const users = await Users.findAll({
      attributes: ["id", "name", "no_hp", "role"],
    });
    res.json(users);
  } catch (error) {
    console.error("Kesalahan getUsers:", error);
    res.status(500).json({ msg: "Terjadi kesalahan pada server" });
  }
};

export const Register = async (req, res) => {
  // 1. Ambil dan validasi data dari body request
  const { name, no_hp, password, confPassword } = req.body;
  if (!name || !no_hp || !password || !confPassword) {
    return res.status(400).json({ msg: "Semua field wajib diisi." });
  }
  if (password !== confPassword) {
    return res
      .status(400)
      .json({ msg: "Password dan konfirmasi password tidak cocok." });
  }

  try {
    // 2. Cek apakah nomor telepon sudah terdaftar
    const userExists = await Users.findOne({ where: { no_hp } });
    if (userExists) {
      return res.status(409).json({ msg: "Nomor telepon sudah terdaftar." });
    }

    // 3. Hash password sebelum disimpan
    const salt = await bcrypt.genSalt();
    const hashPassword = await bcrypt.hash(password, salt);

    // 4. Buat pengguna baru di tabel Users
    const newUser = await Users.create({
      name: name,
      no_hp: no_hp,
      password: hashPassword,
      role: "user", // Default role untuk registrasi baru
    });

    // 5. Buat entri data peserta yang terhubung
    // Fungsionalitas ini dipertahankan dari kode asli Anda
    await DataPeserta.create({
      no_hp: newUser.no_hp,
      name: newUser.name, // Sinkronisasi nama
      nama_peserta: newUser.name, // Sinkronisasi nama peserta
    });

    // 6. Buat Access Token dan Refresh Token
    const payload = {
      userId: newUser.id,
      nohp: newUser.no_hp,
      role: newUser.role,
    };
    const accessToken = jwt.sign(payload, process.env.JWT_SECRET, {
      expiresIn: "15m",
    });
    const refreshToken = jwt.sign(payload, process.env.JWT_REFRESH_SECRET, {
      expiresIn: "7d",
    });

    // 7. Simpan refresh token ke database untuk pengguna yang baru dibuat
    await Users.update(
      { refresh_token: refreshToken },
      {
        where: { id: newUser.id },
      }
    );

    // 8. Kirim token sebagai httpOnly cookies untuk keamanan
    res.cookie("accessToken", accessToken, {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production", // Hanya secure di produksi
      maxAge: 15 * 60 * 1000, // 15 menit
      sameSite: "lax",
    });

    res.cookie("refreshToken", refreshToken, {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production", // Hanya secure di produksi
      maxAge: 7 * 24 * 60 * 60 * 1000, // 7 hari
      sameSite: "lax",
    });

    // 9. Kirim respons sukses
    res.status(201).json({ msg: "Registrasi berhasil" });
  } catch (error) {
    // Tangani kemungkinan error dari server atau database
    console.error("Kesalahan saat proses registrasi:", error);
    res.status(500).json({ msg: "Terjadi kesalahan pada server." });
  }
};

export const Login = async (req, res) => {
  try {
    const { no_hp, password } = req.body;

    console.log("Mencoba login dengan:", { no_hp });
    const user = await Users.findOne({ where: { no_hp } });

    if (!user) {
      console.log("User tidak ditemukan:", { no_hp });
      return res.status(404).json({ msg: "Nomor telepon tidak ditemukan" });
    }

    const match = await bcrypt.compare(password, user.password);
    if (!match) {
      console.log("Password salah untuk:", { no_hp });
      return res.status(400).json({ msg: "Password salah" });
    }

    if (!process.env.JWT_SECRET || !process.env.JWT_REFRESH_SECRET) {
      console.error(
        "Variabel lingkungan JWT_SECRET atau JWT_REFRESH_SECRET tidak diatur"
      );
      return res.status(500).json({ msg: "Konfigurasi server tidak lengkap" });
    }

    const accessToken = jwt.sign(
      { userId: user.id, nohp: user.no_hp, role: user.role },
      process.env.JWT_SECRET,
      { expiresIn: "15m" }
    );
    const refreshToken = jwt.sign(
      { userId: user.id, nohp: user.no_hp, role: user.role },
      process.env.JWT_REFRESH_SECRET,
      { expiresIn: "7d" }
    );

    const updateResult = await Users.update(
      { refresh_token: refreshToken },
      { where: { id: user.id } }
    );
    console.log("Hasil update refresh_token (login):", updateResult);

    res.cookie("accessToken", accessToken, {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      maxAge: 15 * 60 * 1000,
      sameSite: "lax",
      path: "/",
    });
    res.cookie("refreshToken", refreshToken, {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      maxAge: 7 * 24 * 60 * 60 * 1000,
      sameSite: "lax",
      path: "/",
    });

    res.json({ msg: "Login berhasil" });
  } catch (error) {
    console.error("Kesalahan login:", error);
    res.status(500).json({ msg: "Terjadi kesalahan pada server" });
  }
};

export const Logout = async (req, res) => {
  try {
    const refreshToken = req.cookies.refreshToken;
    if (!refreshToken) {
      return res.status(204).json({ msg: "Tidak ada refresh token" });
    }

    await Users.update(
      { refresh_token: null },
      { where: { refresh_token: refreshToken } }
    );

    res.clearCookie("accessToken", {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      sameSite: "lax",
      path: "/",
    });
    res.clearCookie("refreshToken", {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      sameSite: "lax",
      path: "/",
    });

    res.status(200).json({ msg: "Logout berhasil" });
  } catch (error) {
    console.error("Kesalahan logout:", error);
    res.status(500).json({ msg: "Terjadi kesalahan pada server" });
  }
};

export const RefreshToken = async (req, res) => {
  try {
    const refreshToken = req.cookies.refreshToken;
    if (!refreshToken) {
      return res.status(401).json({ msg: "Refresh token tidak ditemukan" });
    }

    const user = await Users.findOne({
      where: { refresh_token: refreshToken },
    });
    if (!user) {
      return res.status(403).json({ msg: "Refresh token tidak valid" });
    }

    if (!process.env.JWT_SECRET || !process.env.JWT_REFRESH_SECRET) {
      console.error(
        "Variabel lingkungan JWT_SECRET atau JWT_REFRESH_SECRET tidak diatur"
      );
      return res.status(500).json({ msg: "Konfigurasi server tidak lengkap" });
    }

    jwt.verify(refreshToken, process.env.JWT_REFRESH_SECRET, (err, decoded) => {
      if (err) {
        return res.status(403).json({ msg: "Refresh token tidak valid" });
      }

      const accessToken = jwt.sign(
        { userId: user.id, nohp: user.no_hp, role: user.role },
        process.env.JWT_SECRET,
        { expiresIn: "15m" }
      );

      res.cookie("accessToken", accessToken, {
        httpOnly: true,
        secure: process.env.NODE_ENV === "production",
        maxAge: 15 * 60 * 1000,
        sameSite: "lax",
        path: "/",
      });

      res.json({ msg: "Access token diperbarui" });
    });
  } catch (error) {
    console.error("Kesalahan refresh token:", error);
    res.status(500).json({ msg: "Terjadi kesalahan pada server" });
  }
};
