import express from "express";
import {
  getUsers,
  Register,
  Login,
  Logout,
  RefreshToken,
} from "../controllers/Users.js";
import verifyToken from "../middleware/VerifyToken.js";
import {
  getProfile,
  updateProfile,
  getEvents,
  getEvent,
  createEvent,
  updateEvent,
  deleteEvent,
  getBurung,
  createBurung,
  deleteBurung,
  getUserProfile, // Fungsi baru yang akan ditambahkan
  getUserDetailsForAdminEdit,
  adminUpdateUser,
  getAllBirdsWithOwners,
  registerBirdToEvent,
  adminDeleteUser,
} from "../controllers/AppController.js";
import {
  getRacesByEvent,
  createRace,
  updateRace,
  deleteRace,
} from "../controllers/RaceController.js";

const router = express.Router();

const cookieAuthMiddleware = (req, res, next) => {
  const accessToken = req.cookies.accessToken;
  if (accessToken) {
    req.headers.authorization = `Bearer ${accessToken}`;
    console.log("Access token added to Authorization header");
  } else {
    console.log("No access token found in cookies");
  }
  next();
};

// Authentication routes
router.get("/users", cookieAuthMiddleware, verifyToken, getUsers);
router.post("/users", Register);
router.post("/login", Login);
router.post("/token", cookieAuthMiddleware, RefreshToken);
router.delete("/logout", cookieAuthMiddleware, Logout);
router.get("/me", cookieAuthMiddleware, verifyToken, getProfile);

// Profile routes
router.get("/profile", cookieAuthMiddleware, verifyToken, getProfile);
router.patch("/profile", cookieAuthMiddleware, verifyToken, updateProfile);

// New route for getting a specific user's profile (for admin)
router.get(
  "/users/:id/profile",
  cookieAuthMiddleware,
  verifyToken,
  getUserProfile
);
router.get(
  "/admin/users/:id/details", // <-- PATH BARU (atau sesuaikan dengan preferensi Anda)
  cookieAuthMiddleware,
  verifyToken,
  getUserDetailsForAdminEdit // <-- KONTROLLER BARU
);
router.put(
  "/admin/users/:id", // <-- PATH BARU UNTUK UPDATE
  cookieAuthMiddleware,
  verifyToken,
  adminUpdateUser // <-- KONTROLLER BARU UNTUK UPDATE
);
router.delete(
  "/admin/users/:id",
  cookieAuthMiddleware,
  verifyToken,
  adminDeleteUser // <-- GUNAKAN CONTROLLER BARU
);

// Event routes dan lainnya tetap sama
router.get("/events", cookieAuthMiddleware, verifyToken, getEvents);
router.get("/events/:id", cookieAuthMiddleware, verifyToken, getEvent);
router.post("/events", cookieAuthMiddleware, verifyToken, createEvent);
router.put("/events/:id", cookieAuthMiddleware, verifyToken, updateEvent);
router.delete("/events/:id", cookieAuthMiddleware, verifyToken, deleteEvent);

router.get("/burung", cookieAuthMiddleware, verifyToken, getBurung);
router.post("/burung", cookieAuthMiddleware, verifyToken, createBurung);
router.delete("/burung/:id", cookieAuthMiddleware, verifyToken, deleteBurung);

// Event Registration routes (New Section)
router.post(
  "/events/register-bird", // Endpoint untuk mendaftarkan burung ke event
  cookieAuthMiddleware,
  verifyToken,
  registerBirdToEvent
);
// Endpoint untuk mendapatkan semua race berdasarkan event ID
router.get(
  "/races/:eventId",
  cookieAuthMiddleware,
  verifyToken,
  getRacesByEvent
);
// Endpoint untuk membuat race baru
router.post("/races", cookieAuthMiddleware, verifyToken, createRace);
// Endpoint untuk memperbarui race
router.put("/races/:id", cookieAuthMiddleware, verifyToken, updateRace);
// Endpoint untuk menghapus race
router.delete("/races/:id", cookieAuthMiddleware, verifyToken, deleteRace);

router.get(
  "/admin/all-birds",
  cookieAuthMiddleware,
  verifyToken,
  getAllBirdsWithOwners
);

export default router;
